/**********************************************************************
// Copyright 2018 Autodesk, Inc.  All rights reserved. 
**********************************************************************/
---------------------------------------------------------
-- This block is for future auto-discovery of security tools. 
-- The following is the global variable name of the singleton instance created by running this script.
--  security_tool_instance_name = "CRP_SecurityTool"
---------------------------------------------------------

---------------------------------------------------------
-- The only method required to be implemented as a public method is:
--   'disable()' - disables the tool and sets the global variable name to undefined
-- The disable method is called by SecurityToolsStartup_instance.display_permission_dialog when disabling protection
-- The tool would be re-enabled by re-running the script.
---------------------------------------------------------

-- CRP persistent global variable and callback removal, cleaning of CRP contaminated startup script files
-- version 1.0 - initial version of script in package file

::CRP_Authorization = true -- turns off changes to scene by CRP script

-- load structure that handles dialogs
if ::SecurityToolsDialogs_instance == undefined do
	filein (pathconfig.appendPath (pathconfig.removePathLeaf (getThisScriptFilename())) "SecurityToolsDialogs.ms")
	
struct CRP_SecurityTool_structdef 
(
	private
	m_callbacks_show_support_to = false, -- whether callbacks.show supports to:<stream>
	m_problem_found_count = 0, -- incremented each time any sort of CRP problem (startup scripts or scene) is found.
	m_problem_fixed_count = 0, -- incremented each time any sort of CRP cleanup (startup scripts or scene) occurs.
	m_perform_cleanup_quiet_mode_default = true, -- default cleanp behavior when in quiet mode
	m_security_tools_CRP_help_topic_id = "idh_security_tools_crp",
	m_CRP_startup_script_files_found, -- the CRP startup script files found by test_for_CRP_startup_scripts
	m_perform_CRP_startup_script_file_cleanup, -- whether to clean up the CRP startup script files
	m_perform_CRP_process_cleanup, -- whether to clean up CRP corrupted scene files
	m_postloadCallbacks_registered = false, -- used for controlling registration of postload callbacks
	m_force_local_quiet_mode= false, -- if true, behave as if in quiet mode
		
	m_perform_cleanup_quiet_mode= m_perform_cleanup_quiet_mode_default, -- cleanup behavior when in quiet mode
	
	m_version_number = "1.0",

	-- things to look for....
	m_searchFor = " CRP_AScript" + " = \"",
	m_searchFor_length = m_searchFor.count,
	m_terminating_string = " setINISetting  (getMAXIniFile()) \" MAXScript \" \" LoadSavePersistentGlobals \" \" 1 \"  ",
	m_terminating_string_offset = 67,
	m_CRP_globals = #(#CRP_AScript, #CRP_BScript),
	m_CRP_callback_ids = #(#ID_CRP_preRenderP, #ID_CRP_filePostOpenP, #ID_CRP_viewportChangeP, #ID_CRP_filePostOpen, #ID_CRP_filePostMerge, #ID_CRP_postImport),
	
	public
	
	m_in_file_open = false, -- used for controlling register of filePostMerge
	m_force_print_to_listener = false, -- whether to display listener output when not in quiet mode

	-- disable this tool - required  method
	fn disable =
	(
		-- unregister callbacks
		callbacks.removeScripts id:#CRP_SecurityTool
		callbacks.removeScripts id:#CRP_SecurityTool_startup
		callbacks.removeScripts id:#CRP_SecurityTool_postload
		-- log info
		local msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SECURITY_TOOLS_DISABLED~
		logsystem.logEntry msg info:true broadcast:true
		if m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener do
			format "-- %\n" msg
		-- set global instance variable to undefined
		 ::CRP_SecurityTool = undefined
	),
	
	-- get the problem found and fixed counts. Used by unit tests
	fn get_problem_found_count = return m_problem_found_count,
	fn get_problem_fixed_count = return m_problem_fixed_count,

	-- reset problem found and fixed counts. Used by unit tests
	fn clear_problem_counts = m_problem_found_count = m_problem_fixed_count = 0,

	-- get the name of this script file. Used by unit tests
	fn getToolFileName =
	(
		return (::getSourceFileName())
	),
	
	-- test environment for CRP 
	fn test_for_CRP_in_startup_scripts =
	(
		m_CRP_startup_script_files_found = #()
		local startup_scripts = getfiles ((getDir #startupScripts) + @"\*.ms")
		for sf in startup_scripts do
		(
			local f = openfile sf
			local found_CRP = (skipToString f m_searchFor) != undefined
			close f
			if found_CRP do 
				append m_CRP_startup_script_files_found sf
		)
		
		if (m_CRP_startup_script_files_found.count != 0) do
		(
			m_problem_found_count += 1
			local msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~FOUND_CRP_STARTUP_SCRIPT_FILE~
			for fname in m_CRP_startup_script_files_found do
				logsystem.logentry (msg + fname) warning:true broadcast:true

			local display_listener_output = m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener

			if display_listener_output do
			(
				msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~FOUND_FOLLOWING_CRP_STARTUP_SCRIPT_FILES~ + "\n"
				for f in m_CRP_startup_script_files_found do
					msg += ("  " + f + "\n")
				format "%" msg
			)
			
			m_perform_CRP_startup_script_file_cleanup = m_perform_cleanup_quiet_mode 
			if not (m_force_local_quiet_mode or GetQuietMode()) then
			(
				m_perform_CRP_startup_script_file_cleanup = SecurityToolsDialogs_instance.display_corruptionFoundInEnv_dialog m_CRP_startup_script_files_found m_security_tools_CRP_help_topic_id "CRP"
				if not m_perform_CRP_startup_script_file_cleanup do
					SecurityToolsDialogs_instance.display_corruptionNotCleanedInEnv_dialog m_security_tools_CRP_help_topic_id
			)
			else if display_listener_output do
			(
				if m_perform_CRP_startup_script_file_cleanup then
					msg += ~CRP_ENV_AUTO_CLEAN_QUIET~ + "\n"
				else
					msg += ~CRP_ENV_NOT_AUTO_CLEAN_QUIET~ + "\n"
				format "%" msg
			)
				
			if m_perform_CRP_startup_script_file_cleanup then
			(
				m_problem_fixed_count += 1
				
				-- collect files where there were errors processing the file
				local error_processing_file = #()
				-- remove the CRP related lines from the files, creating backup of each file
				for fname in m_CRP_startup_script_files_found do
				(
					local f = undefined
					try f = openfile fname mode:"r" -- try to open with read access....
					catch ()
					if f == undefined do
					(
						local ss = stringstream ""
						format ~CRP_ENV_CLEAN_FAILURE_FMT1~ fname to:ss
						msg = ~CRP_SECURITY_TOOL~ + m_version_number + (ss as string)
						logsystem.logentry msg warning:true broadcast:true
						if display_listener_output do
							format "%\n" msg
						append error_processing_file fname
						continue
					)
					local fileContents = #()
					while not eof f do append fileContents (readline f)
					local CRP_start_line = undefined
					for i = 1 to fileContents.count while (CRP_start_line == undefined) do
					(
						local line = fileContents[i]
						if (substring line 1 m_searchFor_length) == m_searchFor do 
							CRP_start_line = i
					)
					if CRP_start_line == undefined do
					(
						local ss = stringstream ""
						format ~CRP_ENV_CLEAN_FAILURE_FMT2~ fname to:ss
						msg = ~CRP_SECURITY_TOOL~ + m_version_number + (ss as string)
						logsystem.logentry msg warning:true broadcast:true
						if display_listener_output do
							format "%\n" msg
						append error_processing_file fname
						close f
						continue
					)
					local CRP_end_line = CRP_start_line+m_terminating_string_offset
					if (CRP_end_line > fileContents.count or fileContents[CRP_end_line] != m_terminating_string) do
					(
						local ss = stringstream ""
						format ~CRP_ENV_CLEAN_FAILURE_FMT3~ fname to:ss
						msg = ~CRP_SECURITY_TOOL~ + m_version_number + (ss as string)
						logsystem.logentry msg warning:true broadcast:true
						if display_listener_output do
							format "%\n" msg
						append error_processing_file fname
						close f
						continue
					)
					if (copyfile fname (fname + ".backup")) do
					(
						msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_ENV_CREATED_BACKUP_COPY_OF~ + fname + " (" + (fname + ".backup") + " )"
						if display_listener_output do
							format "%\n" msg
						logsystem.logentry msg info:true broadcast:true
					)
					-- reopen file with write access, delete contents
					close f
					f = undefined
					try f = openfile fname mode:"wt" -- try to open with write access....
					catch ()
					if f == undefined do
					(
						local ss = stringstream ""
						format ~CRP_ENV_CLEAN_FAILURE_FMT4~ fname to:ss
						msg = ~CRP_SECURITY_TOOL~ + m_version_number + (ss as string)
						logsystem.logentry msg warning:true broadcast:true
						if display_listener_output do
							format "%\n" msg
						append error_processing_file fname
						continue
					)
					for i = 1 to fileContents.count where (i < CRP_start_line or i > CRP_end_line) do
					(
						format "%\n" fileContents[i] to:f
					)
					close f
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_ENV_CLEANED_FILE~ + fname
					logsystem.logentry msg info:true broadcast:true
					if display_listener_output do
						format "%\n" msg
				)
				
				if error_processing_file.count == 0 then
				(
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + " - " + ~CRP_ENV_CLEAN_SUCCESS~
					logsystem.logentry msg info:true broadcast:true
					if display_listener_output do
						format "%\n" msg
					if securitytools != undefined do
						securitytools.LogEventById #EnvironmentCRP #CleanupSucceeded
					
					if not (m_force_local_quiet_mode or GetQuietMode()) then
						SecurityToolsDialogs_instance.display_corruptionCleanedInEnv_dialog m_security_tools_CRP_help_topic_id
				)
				else
				(
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_ENV_CLEAN_FAILURE~
					logsystem.logentry msg warning:true broadcast:true
					if securitytools != undefined do
						securitytools.LogEventById #EnvironmentCRP #CleanupFailed
					
					if m_force_local_quiet_mode or GetQuietMode() then
					(
						local ss = stringstream ""
						format (~CRP_ENV_CLEAN_FAILURE_FMT5~ + "\n") (m_CRP_startup_script_files_found.count - error_processing_file.count) m_CRP_startup_script_files_found.count to:ss
						msg = ss as string
						for f in error_processing_file do
							msg += ("  " + f + "\n")
						msg += ~CRP_ENV_CLEAN_FAILURE_MSG~ + "\n"
						msg += m_security_tools_CRP_help_topic_id
						format "%" msg
					)
					else
						SecurityToolsDialogs_instance.display_corruptionErrorCleaningInEnv_dialog error_processing_file m_security_tools_CRP_help_topic_id
				)
			)
			else
			(
				if (m_CRP_startup_script_files_found.count != 0) do
				(
					if securitytools != undefined do
						securitytools.LogEventById #EnvironmentCRP #CleanupDeclined
					for fname in m_CRP_startup_script_files_found do
					(
						msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_ENV_CLEAN_FAILURE_MSG2~ + fname
						logsystem.logentry msg warning:true broadcast:true
						if display_listener_output do
							format "%\n" msg
					)
				)
			)
		)
	),

	-- test process for CRP corruption
	fn test_for_CRP_process_corruption unregisterPostLoad:false unregisterPostStartup:false doingPreSaveCheck:false saveFileName:"" =
	(
		if unregisterPostLoad do
		(
			callbacks.removeScripts id:#CRP_SecurityTool_postload
			m_postloadCallbacks_registered = false
		)
		if unregisterPostStartup do
			callbacks.removeScripts id:#CRP_SecurityTool_startup
		
		local found_CRP_global = false
		for g in m_CRP_globals while (not found_CRP_global) do found_CRP_global = globalVars.isGlobal g
		
		if found_CRP_global do
		(
			m_problem_found_count += 1
			
			local found_CRP_callbacks = false
			if m_callbacks_show_support_to then
			(
				local ss = stringstream ""
				for id in m_CRP_callback_ids do
					callbacks.show id:id to:ss
				seek ss 0
				while not eof ss and not found_CRP_callbacks do
				(
					local l = readline ss
					if l != "OK" do found_CRP_callbacks = true
				)
				free ss
			)
			else
			(
				-- just assume that the callbacks are present, otherwise we spam listener
				 found_CRP_callbacks = true
			)
			
			if found_CRP_callbacks do
				m_problem_found_count += 1
			
			m_perform_CRP_process_cleanup = false
			local display_listener_output = m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener

			local scene_file_name
			if doingPreSaveCheck then
				scene_file_name = saveFileName
			else
				scene_file_name = maxFilePath + maxFileName
			if scene_file_name == "" do scene_file_name = ~CRP_SCENE_UNNAMED~
			
			local msg
			if doingPreSaveCheck then
				msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_DETECTED_MSG_PRESAVE~ + scene_file_name
			else
				msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_DETECTED_MSG~ + scene_file_name
			logsystem.logentry msg warning:true broadcast:true
			if display_listener_output do
				format "%\n" msg

			if m_force_local_quiet_mode or GetQuietMode() then
			(
				m_perform_CRP_process_cleanup = m_perform_cleanup_quiet_mode
				if m_perform_CRP_process_cleanup then
				(
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_AUTO_CLEAN_QUIET~
				)
				else
				(
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_NOT_AUTO_CLEAN_QUIET~
				)
				logsystem.logentry msg info:true broadcast:true
				format "%\n" msg
			)
			else
			(
				m_perform_CRP_process_cleanup = SecurityToolsDialogs_instance.display_corruptionFoundInScene_dialog m_security_tools_CRP_help_topic_id "CRP"
				if not m_perform_CRP_process_cleanup do
					SecurityToolsDialogs_instance.display_corruptionNotCleanedInScene_dialog m_security_tools_CRP_help_topic_id
			)

			if m_perform_CRP_process_cleanup then
			(
				-- remove the callbacks
				if found_CRP_callbacks do
				(
					m_problem_fixed_count += 1
					for id in m_CRP_callback_ids do
						callbacks.removeScripts id:id 		
				)
				
				-- remove the globals
				m_problem_fixed_count += 1
				for g in m_CRP_globals do 
				(
					if (persistents.isPersistent g) do
						persistents.remove g
					globalVars.remove g
				)
				
				setSaveRequired true
				if securitytools != undefined do
					securitytools.LogEventById (if doingPreSaveCheck then #SceneSaveCRP else #SceneLoadCRP) #CleanupSucceeded
				
				if doingPreSaveCheck then
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_CLEAN_PERFORMED_PRESAVE~ + scene_file_name
				else
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_CLEAN_PERFORMED~ + scene_file_name
				logsystem.logentry msg info:true broadcast:true
				if display_listener_output do
					format "%\n" msg
				if not (m_force_local_quiet_mode or GetQuietMode()) then
					SecurityToolsDialogs_instance.display_corruptionCleanedInScene_dialog m_security_tools_CRP_help_topic_id
			)
			else
			(
				if securitytools != undefined do
					securitytools.LogEventById (if doingPreSaveCheck then #SceneSaveCRP else #SceneLoadCRP) #CleanupDeclined
				if doingPreSaveCheck then
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_CLEAN_NOT_PERFORMED_PRESAVE~ + scene_file_name
				else
					msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SCENE_CLEAN_NOT_PERFORMED~ + scene_file_name
				logsystem.logentry msg warning:true broadcast:true
				if display_listener_output do
					format "%\n" msg
			)
		)
		
		if found_CRP_global do
			test_for_CRP_in_startup_scripts()
	),

	fn test_for_CRP =
	(
		test_for_CRP_process_corruption()
		-- this script may run prior to the CRP infected startup script file, so globals would not yet be present. So we want to test for CRP infected startup script files.
		test_for_CRP_in_startup_scripts()
		ok
	),

	fn register_post_load_check =
	(
		if not m_postloadCallbacks_registered do
		(
			m_postloadCallbacks_registered = true
			 -- #animationRangeChange notification fired after loading mxs persistents, but before the #filePostOpen notification is fired
			callbacks.addScript #animationRangeChange "::CRP_SecurityTool.test_for_CRP_process_corruption unregisterPostLoad:true;::CRP_SecurityTool.m_in_file_open = false" id:#CRP_SecurityTool_postload
			callbacks.addScript #filePostOpen "::CRP_SecurityTool.test_for_CRP_process_corruption unregisterPostLoad:true;::CRP_SecurityTool.m_in_file_open = false" id:#CRP_SecurityTool_postload
			callbacks.addScript #fileOpenFailed "::CRP_SecurityTool.test_for_CRP_process_corruption unregisterPostLoad:true;::CRP_SecurityTool.m_in_file_open = false" id:#CRP_SecurityTool_postload
			if not m_in_file_open do
				callbacks.addScript #filePostMerge "::CRP_SecurityTool.test_for_CRP_process_corruption unregisterPostLoad:true" id:#CRP_SecurityTool_postload
		)
	),
	
	fn maybe_pre_save_check =
	(
		local notificationParam = callbacks.notificationParam()
		if notificationParam[1] != 3 do -- 3 == autobackup
			test_for_CRP_process_corruption doingPreSaveCheck:true saveFileName:(notificationParam[2])
	),
	
	on create do
	(
		::CRP_SecurityTool = this
		local msg = ~CRP_SECURITY_TOOL~ + m_version_number + ~CRP_SECURITY_TOOLS_LOADED~ + "  (" + getThisScriptFilename()  + ")"
		logsystem.logEntry msg info:true broadcast:true
		if m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener do
			format "-- %\n" msg

		if (isproperty systemTools #GetBuildNumber) do
		(
			local buildNumberVals= filterstring (systemTools.GetBuildNumber()) "." --> #("20", "4", "0", "0")
			for i = 1 to buildNumberVals.count do buildNumberVals[i] = buildNumberVals[i] as integer
			m_callbacks_show_support_to = buildNumberVals[1] > 20
			if not m_callbacks_show_support_to do
				m_callbacks_show_support_to = buildNumberVals[1] == 20 and (buildNumberVals[4] > 4250 or buildNumberVals[4] == 0)
		)
		
		test_for_CRP()
		
		callbacks.removeScripts id:#CRP_SecurityTool
		callbacks.removeScripts id:#CRP_SecurityTool_startup
		callbacks.removeScripts id:#CRP_SecurityTool_postload
		callbacks.addScript #mtlLibPostOpen "::CRP_SecurityTool.test_for_CRP_process_corruption unregisterPostStartup:true" id:#CRP_SecurityTool_startup -- ensures run at least once at startup after all startup scripts are loaded
		callbacks.addScript #filePreOpen "::CRP_SecurityTool.m_in_file_open = true;::CRP_SecurityTool.register_post_load_check()" id:#CRP_SecurityTool
		callbacks.addScript #filePreMerge "::CRP_SecurityTool.register_post_load_check()" id:#CRP_SecurityTool
		callbacks.addScript #filePreSaveProcess "::CRP_SecurityTool.maybe_pre_save_check()" id:#CRP_SecurityTool
	)
)

(
	CRP_SecurityTool_structdef()
	silentValue
)

-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQBLzs2H7NODorIDq3pKGi1eVe53sZHqTKIF
-- QM/xg0vdMQ+9Q/8eLtDIXF4PDEAkdkhIIBscNgQIQxsX2PZWzS2N4pZq5wiOgV4g
-- j1aS5Jw4/jVtSSsA2/1ffYSbSiNONrA81qSJlol7JkS0VjnAgH+d2MKHHuyYsiC9
-- mJZFGAw/DHTFXUqA8X+zUdmnQfW6MUeWY5WVljAs7NjUGQYHch84+KIk7K9OGnk/
-- rkwujQM3t2TXaimXV5gBtLifjrci4+EVaL1BIbf3hZSMqmygKJVC0RgNDYsH/Wze
-- vQSuzyYbkwMIgDyZePDmt2eCcfaX42l/Si7fyvUDAcXE9PpZAKuK
-- -----END-SIGNATURE-----